/*
 * @brief SCTimer/PWM (SCT) RC5 encoder (transmitter) example
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"


/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
#define RC5_DATA0()      Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 0, IOCON_MODE_PULLDOWN);	// input low (pull down)
#define RC5_DATA1()      Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 0, IOCON_MODE_PULLUP); 		// input high (pull up)

static uint8_t  state;                                 // driver state
static uint8_t  count;                                 // bit counter
static uint8_t  flag;                                  // frame send ready flag
static uint16_t frame;                                 // frame

/*****************************************************************************
 * Private functions
 ****************************************************************************/
void RC5_Init(void);
void RC5_Send(uint16_t fr);

/*****************************************************************************
 * Public functions
 ****************************************************************************/

void RC5_Send(uint16_t fr)
{
    frame = fr << 2;                                   												// MSB first, 14 bits shift left 2
    state = 0;
    flag  = 0;                                         												// clear ready flag
    count = 14;                                        												// set bit count to 14

    Chip_SCT_EnableEventInt(LPC_SCT , SCT_EVT_4 );    												// event 4 generates interrupt

    while (!flag) ;                                    												// wait for ready flag
}

void SCT0_IRQHandler(void)
{
    if (count)                                         												// send next bit if required
    {
        if (state)
        {
            if (frame & 0x8000)
            {
                RC5_DATA1();
            }
            else
            {
                RC5_DATA0();
            }
            state = 0;
            count --;
            frame <<= 1;
        }
        else
        {
            if (frame & 0x8000)
            {
                RC5_DATA0();
            }
            else
            {
                RC5_DATA1();
            }
            state = 1;
        }
    }
    else
    {
        RC5_DATA0();                                       												// inactive carrier pulse

        Chip_SCT_DisableEventInt(LPC_SCT, (CHIP_SCT_EVENT_T) 0x00);								// disable interrupt
        flag = 1;                                          												// signal ready
    }
    Chip_SCT_SetEventFlag(LPC_SCT, SCT_EVT_4);																		// clear event 4 flag
}

void RC5_Init(void)
{
    RC5_DATA0();                                           												// SCT0_IN0 low

    Chip_SCT_Init(LPC_SCT);																												// enable the SCT clock

//    Chip_SCT_Config(LPC_SCT, SCT_CONFIG_32BIT_COUNTER |
//    						  SCT_CONFIG_AUTOLIMIT_L   );																		// unified timer and auto limit


    Chip_GPIO_SetPinDIR(LPC_GPIO, 0, 24, false);

    Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 24, 	IOCON_FUNC0 		  	|
                                             IOCON_I2C_SLEW			|
                                            IOCON_DIGITAL_EN		|
                                              IOCON_INPFILT_OFF		|
                                             0x0 << 6			    |
                                             0x0 << 9			    |
                                              0x1 << 10			    );								// SCT_IN1  = P0.24 = SW1

    LPC_SCT->OUTPUT      = (1 << 0);                                                                                                    // SCT0_OUT0 (IR output) is high

    Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 7, IOCON_FUNC2| IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);
    


/********************************************************************
*  SCT_L: low part configuration:
********************************************************************/

//    Chip_SCT_SetMatchCountL(LPC_SCT, SCT_MATCH_0, (SystemCoreClock / 48000)-1);		// 75% of 36 KHz
    Chip_SCT_SetMatchReloadL(LPC_SCT, SCT_MATCH_0, (SystemCoreClock / 48000)-1);    // 75% of 36 KHz

//    Chip_SCT_SetMatchCountL(LPC_SCT, SCT_MATCH_1, (SystemCoreClock / 36000)-1);		// 36 KHz
    Chip_SCT_SetMatchReloadL(LPC_SCT, SCT_MATCH_1, (SystemCoreClock / 36000)-1);

    Chip_SCT_EventStateMask(LPC_SCT, 0, ENABLE_ALL_STATES);													// event 0 happens in all states
    Chip_SCT_EventControl(LPC_SCT, 0, (CHIP_SCT_EVENTCTRL_T) (SCT_EVECTRL_MATCH0	|	// MATCHSEL [3:0]   = related to match 0
                                                              SCT_HEVENT_L		    |	// HEVENT   [4]     = use L state & match
                                                               SCT_OUTSEL_L		    |	// OUTSEL   [5]     = select input from IOSEL
                                                                SCT_IOSEL_1	  		|	// IOSEL    [9:6]   = select input 1
                                                              SCT_IOCOND_HIGH	    |	// IOCOND   [11:10] = high level
                                                             SCT_COMBMODE_AND	    |	// COMBMODE [13:12] = match AND IO condition
                                                              SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                              SCT_STATEEV_0		    |	// STATEV   [19:15] = no state change
                                                               SCT_MATCHMEM		    |	// MATCHMEM [20]    = "equal" to match
																														SCT_DIRECTION		    ));	// DIRECTION[22:21] = direction independent


    Chip_SCT_EventStateMask(LPC_SCT, 1, ENABLE_ALL_STATES);													// event 1 happens in all states
    Chip_SCT_EventControl(LPC_SCT, 1, (CHIP_SCT_EVENTCTRL_T) (SCT_EVECTRL_MATCH1	|	// MATCHSEL [3:0]   = related to match 1
                                                               SCT_COMBMODE_MATCH	|	// COMBMODE [13:12] = match condition only
																																	SCT_STATELD_0		|	// STATELD  [14]    = STATEV is added to state
                                                                SCT_STATEEV_0		));	// STATEV   [19:15] = no state change


    Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_0);													// IR LED low  @ event 0
    Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_1);														// IR LED high @ event 1

    LPC_SCT->LIMIT_L        = (1 << 1);                   													// events 1 is used as counter limit
    

/********************************************************************
*  SCT_H: high part configuration:
********************************************************************/

 //   Chip_SCT_SetMatchCountH(LPC_SCT, SCT_MATCH_0, (SystemCoreClock / 1125)-1);		// 36 KHz / 32 ticks = 1125
    Chip_SCT_SetMatchReloadH(LPC_SCT, SCT_MATCH_0, (SystemCoreClock / 1125)-1);


    Chip_SCT_EventStateMask(LPC_SCT, 4, ENABLE_ALL_STATES);													// event 4 only happens in all states
    Chip_SCT_EventControl(LPC_SCT, 4, (CHIP_SCT_EVENTCTRL_T) (SCT_EVECTRL_MATCH0	|	// MATCHSEL [3:0]   = related to match_H 0
                                                              SCT_HEVENT_H		    |	// HEVENT   [4]     = use H state & match
                                                              SCT_COMBMODE_MATCH	|	// COMBMODE [13:12] = match condition only
                                                              SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
																														SCT_STATEEV_0		    ));	// STATEV   [19:15] = no state change


    LPC_SCT->LIMIT_H        = (1 << 4);                   													// event 4 is used as H counter limit

    NVIC_EnableIRQ(SCT0_IRQn);                             													// enable SCT interrupt

    Chip_SCT_ClearControl(LPC_SCT,(SCT_CTRL_HALT_L |SCT_CTRL_HALT_H));      				// unhalt both low and high counter

}


void MRT_IRQHandler(void)
{
	Chip_MRT_IntClear(LPC_MRT_CH0);                 																	// acknowledge MRT irq
}

void MRT_Init(void)
{
	Chip_MRT_Init();									  																							// enable MRT clock

	Chip_MRT_SetInterval(LPC_MRT_CH0, (SystemCoreClock / 8)	|
									  	  	  (1U << 31)	);																					// 1/8 sec (125 msec) timeout


	Chip_MRT_SetMode(LPC_MRT_CH0, 	(MRT_MODE_T)  ( (0 << 1)|
                                    			MRT_MODE_REPEAT	));     									// enable + repeated mode
    NVIC_EnableIRQ(MRT_IRQn);                              													// enable Multi-Rate timer irq
}



int main(void)
{
  static uint8_t  trg = 1;
  static uint8_t  tgl = 1;
  static uint16_t frame;

  SystemCoreClockUpdate();
  Board_Init();

  Chip_Clock_SetMain_A_ClockSource(SYSCON_MAIN_A_CLKSRC_FRO12MHZ);									// Set Main Clock (SCT Clock) as 12MHz

  Chip_Clock_EnablePeriphClock((CHIP_SYSCON_CLOCK_T) ( SYSCON_CLOCK_GPIO0	|    			// enable GPIO port 0 clock
																											SYSCON_CLOCK_GPIO1	| 	      // enable GPIO port 1 clock
																											SYSCON_CLOCK_IOCON	|         //enable IOCON clock
																										SYSCON_CLOCK_INPUTMUX	));				//enable MUX clock


  MRT_Init();
  RC5_Init();

    while (1)
    {
        frame = (1   << 13) |                              													// set start (S) bit
                (1   << 12) |                              													// set field (F) bit
                (tgl << 11) |                             	 												// clr control / toggle (C) bit
                (20  << 6 ) |                              													// system  = 20
                (9   << 0 );                               													// command = 9

        if (Chip_GPIO_GetPinState(LPC_GPIO, 0, 24))                									// P0_24 high to low?
        {
            if (trg)
            {
                trg = 0;                                   													// trigger P0_24 low
            }
            RC5_Send(frame);
        }
        else if ((Chip_GPIO_GetPinState(LPC_GPIO,0, 24)) && !trg)    								// P0_24 low to high?
        {
            trg = 1;                                       													// trigger P0_24 high
            tgl = !tgl;                                    													// toggle RC5 control bit
        }
        __WFI();
    }
}
